/***************************************************************************/
/*  Copyright 1996 National Instruments Corporation.  All Rights Reserved. */
/***************************************************************************/

#include <utility.h>
#include <stdio.h>
#include <stdlib.h>
#include <formatio.h>
#include <visa.h>
#include "mxox802x.h"

#define mxOX802x_REVISION   "Rev 2.0, 02/1999, CVI 5.0.1" /* Instrument driver revision */
#define BUFFER_SIZE          512L   /* File I/O buffer size              */
#define TRACE_BUFFER_SIZE   9000L   /* Trace input buffer size           */
#define MODE_ANALOG            4    /* Analog mode setting value         */
#define CHANNEL_TWO            2    /* Channel 2 setting value           */
#define MANUAL_PRETRG_DELAY   -1    /* Manual delay setting value        */
#define SINGLE_SHOT            1    /* Single shot setting value         */
#define DEFAULT_TMO            0.1  /* Default timeout in t/div          */
#define EXPANSION_NO_CHANGE    10   /* Expansion no change setting value */ 

/*=Metrix OX8020/8027 Oscilloscopes =========================================*/
/* LabWindows/CVI 5.0.1 Instrument Driver                                    */
/* Original Release: October 1996                                            */
/* By: Jan Sima, Technical University Ostrava, Czech Republic                */
/*     PH. +42(0)696994881  Fax  +42(0)696919597                             */
/*     e-mail: Jan.Sima@vsb.cz                                               */
/*                                                                           */
/* Metrix Technical Support: e-mail: Metrix@dial.oleane.com                  */
/*                           PH. +33(0)450 64 22 22  Fax  +33(0)450 64 22 99 */
/*                                                                           */
/* For LabWindows/CVI Technical Support in the United States call:           */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 683-5678                      */
/*                                                                           */
/* Modification History: None                                                */
/*       																	 */
/*      10/01/96 - Instrument Driver Created.                  				 */
/*       																	 */
/*      02/01/99 - Added OX8020 and RS-232 communication                     */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
static ViString acqModeStr[] = {"REFR;:ACQ:PEAK OFF;","ROLL;","SING;","REFR;:ACQ:PEAK ON;"}; 
static ViString swOnOff[] = {"OFF","ON",""};
static ViString inpCoupling[] = {":COUP AC;",":COUP DC;",":COUP GND;"};
static ViString inpRange[] = {"1E-3;","2E-3;","5E-3;","10E-3;","20E-3;","50E-3;","100E-3;","200E-3;","500E-3;","1;","2;","5;","10;","20;"};
static ViString timeBase[] = {"0.1E-6","0.2E-6","0.5E-6","1E-6","2E-6","5E-6","10E-6","20E-6","50E-6","0.1E-3","0.2E-3","0.5E-3","1E-3","2E-3","5E-3","10E-3","20E-3","50E-3","100E-3","200E-3","500E-3","1","2","5","10","20","50","100","200"};
static ViString expHoriz[] = {";:TIM:EXPH OFF",";:TIM:EXPH ON",""};
static ViString dispModeStr[] = {"CH1;","ALT;","CHOP;","CH2;","XY;","ADD;","MULT;"};
static ViString dispFltStr[] = {":DISP:FILT OFF",":DISP:FILT LN",":DISP:FILT SN"};
static ViString dispTracFeed[] = {"A","B","AB","BA"};
static ViString trgModeStr[] = {":TIM:MODE NORM;:TRIG:MODE NORM;",":TIM:MODE NORM;:TRIG:MODE PTP;",":TRIG:MODE NORM;:TIM:MODE AUTO;"};
static ViString trgSrcStr[] = {"CH1;","CH2;","ALT;","LINE;","EXT;"};
static ViString trgCplStr[] = {"DC;","AC;","LF;","HF;","TVH;","TVV;"};
static ViString trgSlopStr[] = {"-","+"};
static ViString trgCmdWStr[] = {":INIT:CONT ON","*CLS;*TRG;*WAI;*OPC?",":ABOR"};
static ViString trgCmdStr[] = {":INIT:CONT ON","*CLS;*TRG",":ABOR"};
static ViString measFuncStr[] = {"AC","DC","FREQ","PER","PHAS","AMPL","LOW","HIGH","RTIME","FTIME","NWID","PWID","PDUTY","NDUTY","MIN","MAX","PTP"};
static ViString tracStr[] = {"CH1A","CH1B","CH2A","CH2B","S1","S2","CH1AL","CH1AH","CH1BL","CH1BH","CH2AL","CH2AH","CH2BL","CH2BH"};
static ViString tracStr2[] = {"CH1","CH2","S1","S2","CH1L","CH1H","CH2L","CH2H"};

/*****************************************************************************/
/*= GLOBAL VARIABLE DECLARATIONS ============================================*/
/*****************************************************************************/
ViByte traceBuffer[TRACE_BUFFER_SIZE];      /* Raw trace data buffer */ 

/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* mxOX802x_stringValPair is used in the mxOX802x_errorMessage function      */
/*===========================================================================*/
typedef struct  mxOX802x_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  mxOX802x_tStringValPair;


struct mxOX802x_statusDataRanges {
    ViInt16 model;
};

typedef struct mxOX802x_statusDataRanges *mxOX802x_instrRange;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mxOX802x_invalidViBooleanRange (ViBoolean val);
ViBoolean mxOX802x_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mxOX802x_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus mxOX802x_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus mxOX802x_defaultInstrSetup (ViSession openInstrSession);
ViStatus mxOX802x_instrStatus (ViSession instrSession);

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_init (ViRsrc resourceName, ViBoolean IDQuery,
                    ViBoolean resetDevice, ViPSession instrSession)
{
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViSession rmSession = 0;
    ViUInt32 retCnt = 0;
    ViByte rdBuffer[BUFFER_SIZE];
	ViUInt16 interface;                         

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViBooleanRange (IDQuery))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViBooleanRange (resetDevice))
        return VI_ERROR_PARAMETER3;

    /*- Open instrument session ---------------------------------------------*/
    if ((mxOX802x_status = viOpenDefaultRM (&rmSession)) < 0)
        return mxOX802x_status;

    if ((mxOX802x_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) {
        viClose (rmSession);
        return mxOX802x_status;
    }

    viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &interface);
    if(interface==VI_INTF_ASRL) {
        viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 9600);
        viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_RTS_CTS);
        viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0A);
        viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN,VI_ASRL_END_TERMCHAR);
    }

    /*- Configure VISA Formatted I/O ----------------------------------------*/
    if ((mxOX802x_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 10000)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
    if ((mxOX802x_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
    if ((mxOX802x_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
    if ((mxOX802x_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);

    /*- Identification Query ------------------------------------------------*/
    if (IDQuery) {
        /*- Model Identification ------------------------------------------------*/
        if ((mxOX802x_status = viWrite (*instrSession, "\n*IDN?\n", 7, &retCnt)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
        if ((mxOX802x_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
            return mxOX802x_status;
        if (FindPattern (rdBuffer, 0, -1, "OX8020", 0, 0) == -1 && FindPattern (rdBuffer, 0, -1, "OX8027", 0, 0) == -1) 
            return mxOX802x_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
	}

    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice) {
        if ((mxOX802x_status = mxOX802x_reset (*instrSession)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
    }       
    else {
        /*- Send Default Instrument Setup ---------------------------------*/
        if ((mxOX802x_status = mxOX802x_defaultInstrSetup (*instrSession)) < 0)
            return mxOX802x_initCleanUp (rmSession, instrSession, mxOX802x_status);
    }
    
    if ((mxOX802x_status = mxOX802x_instrStatus (*instrSession)) < 0)
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functionss use.    */
/*           It works as a simple dual channel digitizer.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_example (ViSession instrSession, ViInt32 timebase,
                           ViInt32 voltageRange, ViInt32 coupling, ViReal64 _VI_FAR data[],
                           ViPReal64 deltaT, ViPInt32 numberofSamples)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViReal64 t0;
    
    /*- Check input parameter ranges ----------------------------------------*/    
    if (mxOX802x_invalidViInt32Range (timebase, 0, 19))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (voltageRange, 0, 13))
        return VI_ERROR_PARAMETER3;
    if (mxOX802x_invalidViInt32Range (coupling, 0, 2))
        return VI_ERROR_PARAMETER4;
    
    /*- Resetting the instrument into a known state -------------------------*/
    if((mxOX802x_status = mxOX802x_reset (instrSession)) < 0)
        return mxOX802x_status;

    /*- Configuring single-shot digital acquisition -------------------------*/
    if((mxOX802x_status = mxOX802x_confAcq (instrSession, 2, VI_OFF)) < 0)
        return mxOX802x_status;
    
    /*- Setting display mode to CH1 -----------------------------------------*/     
    if((mxOX802x_status = mxOX802x_confDisp (instrSession, 0, 0, 0, 0)) < 0)
        return mxOX802x_status;
        
    /*- Configuring voltage input range and coupling ------------------------*/ 
    if((mxOX802x_status = mxOX802x_confInp (instrSession, 1, voltageRange, coupling, VI_OFF)) < 0)
        return mxOX802x_status;
    
    /*- Setting timebase ----------------------------------------------------*/
    if((mxOX802x_status = mxOX802x_confHoriz (instrSession, timebase, 10)) < 0)
        return mxOX802x_status;
    
    /*- Setting trigger mode to AUTO, to make sure the instrument starts ----*/
    /*- acquiring data immediatelly after it is triggered.-------------------*/
    if((mxOX802x_status = mxOX802x_confTrig (instrSession, 0, 2, 0, VI_ON, 8)) < 0)
        return mxOX802x_status;
        
    /*- Sending single-shot trigger in wait mode. This function will not ----*/
    /*- release control as long as the acquisition is running.---------------*/
    if((mxOX802x_status = mxOX802x_trigger (instrSession, 1, VI_ON)) < 0)
        return mxOX802x_status;
        
    /*- Reads trace S1 from the instrument ----------------------------------*/ 
    if((mxOX802x_status = mxOX802x_readTrace (instrSession, 4, data, numberofSamples, &t0, deltaT)) < 0)
        return mxOX802x_status; 
                               
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Read/Write Setup                                                */
/* Purpose:  This function reads/writes the instrument setup data from/to    */
/*           the instrument. The setup data being read from the instrument   */
/*           always contains current configuration.                          */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_readWriteSetup (ViSession instrSession, ViBoolean mode,
                                  ViChar _VI_FAR instrumentSetup[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViBooleanRange (mode))
        return VI_ERROR_PARAMETER2;
    
    if (mode == VI_TRUE) {
        /*- Command to read instrument setup --------------------------------*/
        Fmt (comBuffer, "%s<*LRN?\n");
    }
    else {
        /*- Command to write setup to the instrument ------------------------*/
        Fmt (comBuffer, "%s<:SYST:SET %s[t-w57]\n", instrumentSetup);
    }
    
    if((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    if (mode == VI_TRUE) {  
        /*- Reading instrument setup ----------------------------------------*/
        if ((mxOX802x_status = viRead (instrSession, instrumentSetup, 57, &retCnt)) < 0)
            return mxOX802x_status;
    }
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
        
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Autoscale                                                       */
/* Purpose:  This function initiates the autoscaling operation. It waits     */
/*           until the autoscaling operation is completed.                   */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_autoScale (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           

    /*- Command to run autosetup and wait for its completion ----------------*/
    Fmt (comBuffer, "%s<:AUT;*OPC?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    /*- Reading operation complete query response after autosetup has completed --*/        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Acquisition                                              */
/* Purpose:  This function configures the acquisition modes of the           */ 
/*           instrument. The following settings are accessible: acquisition  */
/*           mode incl. envelope mode, glitch detect function.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confAcq (ViSession instrSession, ViInt32 mode,
                           ViBoolean glitchDetect)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           
    mxOX802x_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/ 
    if (mxOX802x_invalidViInt32Range (mode, 0, 4))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViBooleanRange (glitchDetect))
        return VI_ERROR_PARAMETER3;
    
    if (mode == MODE_ANALOG) {
        /*- Command to switch to analog mode --------------------------------*/
        Fmt (comBuffer, "%s<:MODE AN\n");
    }
    else {
	    viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);
	    
        /*- Command to control digital acquisition modes --------------------*/
        if(instrPtr -> model == 1)
            Fmt (comBuffer, "%s<:MODE NUM;:ACQ:MODE %s:ACQ:GLIT %s\n", acqModeStr[mode], swOnOff[glitchDetect]);
        else
            Fmt (comBuffer, "%s<:MODE NUM;:ACQ:MODE %s\n", acqModeStr[mode]);
    }
        
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
        
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Input                                                    */
/* Purpose:  This function configures the selected input channel of the      */
/*           instrument. The following settings are accessible: channel,     */
/*           coupling, range, second channel inversion switch.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confInp (ViSession instrSession, ViInt32 channel,
                           ViInt32 vrange, ViInt32 coupling, ViBoolean invertCH2)
{
    ViUInt32 retCnt = 0;
    ViUInt32 byteCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (channel, 1, 2))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (vrange, 0, 13))
        return VI_ERROR_PARAMETER3;
    if (mxOX802x_invalidViInt32Range (coupling, 0, 2))
        return VI_ERROR_PARAMETER4;
    if (mxOX802x_invalidViBooleanRange (invertCH2))
        return VI_ERROR_PARAMETER5;
    
    /*- Command to configure selected channel's input range and coupling ----*/
    Fmt (comBuffer, "%s<:INP%d[b4]%sRANG %s", channel, inpCoupling[coupling], inpRange[vrange]);
    byteCnt = NumFmtdBytes ();
    if (channel == CHANNEL_TWO) {
        /*- Command to switch CH2 inversion ---------------------------------*/
        Fmt (comBuffer, "%s[a]<INV %s\n", swOnOff[invertCH2]);
    }
    else {
        Fmt (comBuffer, "%s[a]<\n");
    }   
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes () + byteCnt, &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
        
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Horizontal                                               */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument. The following settings are available: range         */
/*           (the actual timebase), horizontal expansion.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confHoriz (ViSession instrSession, ViInt32 hrange,
                             ViInt32 expansion)
{
    ViUInt32 retCnt = 0;
    ViUInt32 byteCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           
    mxOX802x_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (hrange, 0, 28))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (expansion, 0, 10))
        return VI_ERROR_PARAMETER3;
    
	viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);
    
    if(instrPtr -> model == 0 && mxOX802x_invalidViInt32Range (expansion, 1, 10))
        return VI_ERROR_PARAMETER3;
        
    /*- Determining whether the acquisition is running or not ---------------*/
    Fmt (comBuffer, "%s<:INIT:CONT?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;
        
    Scan (comBuffer, "OFF");
    retCnt = NumFmtdBytes ();

    if (retCnt != 3 && expansion != EXPANSION_NO_CHANGE)
        return MXOX802X_ERROR_INSTR_SET_EXPANSION;

    Fmt (comBuffer, "%s<:TIM:RANG %s", timeBase[hrange]);
    byteCnt = NumFmtdBytes ();
    
    if (retCnt == 3) {
        /*- Acquisition is not running, setting expansion -------------------*/
        if (expansion < 8) {
            Fmt (comBuffer, "%s[a]<;:DISP:TRAC:X:PDIV %d[b4]\n", expansion);
        }   
        else {
            Fmt (comBuffer, "%s[a]<%s\n", expHoriz[expansion - 8]);
        }   
    }
    else {
        Fmt (comBuffer, "%s[a]<\n");
    }   
    
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes () + byteCnt, &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Save Trace                                               */
/* Purpose:  This function controls the trace memories. The trace memories   */
/*           can be either filled with the trace data or flushed.            */ 
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confSaveTrace (ViSession instrSession, ViInt32 trace,
                                 ViBoolean mode)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];    
    mxOX802x_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (trace, 0, 3))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViBooleanRange (mode))
        return VI_ERROR_PARAMETER3;
    
	viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);

    if(instrPtr -> model == 0) {
        if (trace == 2) 
            trace = 1;
        else if (trace != 0)
            return VI_ERROR_PARAMETER2;
    }
    
    /*- Determining whether the acquisition is running or not ---------------*/
    Fmt (comBuffer, "%s<:INIT:CONT?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;

    Scan (comBuffer, "OFF");
    retCnt = NumFmtdBytes ();
    
    if (retCnt == 3) {
        /*- Acquisition is not running, saving/flushing traces --------------*/
        if(instrPtr -> model == 0) {
            Fmt (comBuffer, "%s<:DISP:SAVE%d[b4] %s\n", ++trace, swOnOff[mode]);
        }
        else
            Fmt (comBuffer, "%s<:DISP:SAVE%d[b4] %s\n", ++trace, swOnOff[mode]);
        
        if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
            return mxOX802x_status;
    }       
    else {
        return MXOX802X_ERROR_INSTR_SAVE_TRACE;
    }   
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Display                                                  */
/* Purpose:  This function performs the instrument's display configuration.  */
/*           The following settings are accessible via this function:display */
/*           mode, filtering (digital interpolation), trace feed (what       */
/*           traces are going to be displayed on the screen), screen window  */
/*           origin.                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confDisp (ViSession instrSession, ViInt32 displayMode,
                            ViInt32 feedTraces, ViInt32 interpolation,
                            ViInt32  windowOrigin)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (displayMode, 0, 6))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (feedTraces, 0, 3))
        return VI_ERROR_PARAMETER3;
    if (mxOX802x_invalidViInt32Range (interpolation, 0, 2))
        return VI_ERROR_PARAMETER4;
    if (mxOX802x_invalidViInt32Range (windowOrigin, 0, 8000))
        return VI_ERROR_PARAMETER5;

    /*- Command to set interpolation off ------------------------------------*/
    Fmt (comBuffer, "%s<:DISP:FILT OFF\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    /*- Command to set display mode, trace feed and position ----------------*/
    Fmt (comBuffer, "%s<:DISP:MODE %s;:DISP:TRAC:FEED %s;:DISP:TRAC:X:LEFT %d[b4]\n", 
    dispModeStr[displayMode], dispTracFeed[feedTraces], windowOrigin);
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    /*- Command to set the desired interpolation method ---------------------*/
    Fmt (comBuffer, "%s<%s\n", dispFltStr[interpolation]);
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Config Trigger                                                  */
/* Purpose:  This function configures the triggering system of the           */
/*           instrument. The following settings can be configured with this  */
/*           function: trigger source, trigger mode, coupling, trigger slope,*/
/*           pretrigger or delay.                                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_confTrig (ViSession instrSession, ViInt32 source,
                            ViInt32 mode, ViInt32 coupling, ViBoolean slope,
                            ViInt32 pretriggerDelay)
{
    ViUInt32 retCnt = 0;
    ViUInt32 byteCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           
    mxOX802x_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (source, 0, 4))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (mode, 0, 2))
        return VI_ERROR_PARAMETER3;
    if (mxOX802x_invalidViInt32Range (coupling, 0, 5))
        return VI_ERROR_PARAMETER4;
    if (mxOX802x_invalidViBooleanRange (slope))
        return VI_ERROR_PARAMETER5;
    if (mxOX802x_invalidViInt32Range (pretriggerDelay, -1, 8))
        return VI_ERROR_PARAMETER6;

	viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);

    if(instrPtr -> model == 0 && mxOX802x_invalidViInt32Range (pretriggerDelay, -1, 4))
        return VI_ERROR_PARAMETER6;

    /*- Command to set trigger mode, source, coupling and slope -------------*/
    Fmt (comBuffer, "%s<%s:TRIG:SOUR %s:TRIG:COUP %s:TRIG:SLOP %s;", trgModeStr[mode], trgSrcStr[source], trgCplStr[coupling], trgSlopStr[slope]);
    byteCnt = NumFmtdBytes ();
    if (pretriggerDelay == MANUAL_PRETRG_DELAY) {
        /*- Command to set trigger delay manually ---------------------------*/
        Fmt (comBuffer, "%s[a]<:ACQ:REF OFF;:TIM:DEL DEL\n");
    }
    else {
        /*- Command to set pretrigger ---------------------------------------*/
        Fmt (comBuffer, "%s[a]<:TIM:DEL OFF;:ACQ:REF %d[b4]K\n", pretriggerDelay);      
    }
    
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes () + byteCnt, &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
        
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Trigger                                                         */
/* Purpose:  This function controls the triggering system of the instrument. */
/*           Triggering the oscilloscope runs the acquisition.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_trigger (ViSession instrSession, ViInt32 mode, ViBoolean wait)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];  
    ViReal64 timePDiv;

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (mode, 0, 2))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViBooleanRange (wait))
        return VI_ERROR_PARAMETER3;

    if (mode == SINGLE_SHOT) {
        /*- Switchnig to single mode if necessary ---------------------------*/
        Fmt (comBuffer, "%s<:ACQ:MODE?\n");
        if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
            return mxOX802x_status;
        
        if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
            return mxOX802x_status;

        Scan (comBuffer, "REFR");
        if (NumFmtdBytes () == 4) {
            Fmt (comBuffer, "%s<:ACQ:MODE SING\n");
            if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
                return mxOX802x_status;
        }
    
        if (wait == VI_TRUE) {
            /*- Waiting for the acquisition to complete ---------------------*/
            
            /*- Determining the time the acquisition is going to take -------*/
            Fmt (comBuffer, "%s<:TIM:RANG?\n");
            if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
                return mxOX802x_status;
            if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
                return mxOX802x_status;
            if (Scan (comBuffer, "%f", &timePDiv) != 1) 
                return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
            if (timePDiv > DEFAULT_TMO) {
                /*- Increasing the timeout if the estimated acquisition time */
                /*- is longer than VISA timeout -----------------------------*/
                if ((viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, RoundRealToNearestInteger (100000.0*timePDiv))) < 0)
                    return mxOX802x_status;
            }       
        
            /*- Command to trigger the measurement and wait for operation ---*/
            /*- complete query response -------------------------------------*/
            Fmt (comBuffer, "%s<%s\n", trgCmdWStr[SINGLE_SHOT]);
        }
        else {
            /*- Command to trigger the measurement with Wait off ------------*/
            Fmt (comBuffer, "%s<%s\n", trgCmdStr[SINGLE_SHOT]);
        }
    }   
    else {
        /*- Command to trigger the measurement with Wait off ----------------*/
        Fmt (comBuffer, "%s<%s\n", trgCmdStr[mode]);
    }
    
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if (mode == SINGLE_SHOT && wait == VI_TRUE) {
        /*- Reading the operation complete query response -------------------*/
        if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
            return mxOX802x_status;
        /*- Restoring the timeout to the default value of 10000 msecs -------*/
        if ((mxOX802x_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 10000)) < 0)
            return mxOX802x_status;
    }       
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
     
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Measure                                                         */
/* Purpose:  This function forces the instrument to perform selected         */
/*           measurement and returns the result.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_measure (ViSession instrSession, ViInt32 function,
                           ViInt32 source, ViPReal64 output)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           
    mxOX802x_instrRange instrPtr;

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (function, 0, 16))
        return VI_ERROR_PARAMETER2;
    if (mxOX802x_invalidViInt32Range (source, 0, 3))
        return VI_ERROR_PARAMETER3;

	viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);

    if(instrPtr -> model == 0) {
        if (source == 2) source = 1;
        else if (source != 0)
            return VI_ERROR_PARAMETER2;
    }

    /*- Command to take a specified measurement on selected channel ---------*/
    Fmt (comBuffer, "%s<:MEAS:%s? (@%d[b4])\n", measFuncStr[function], ++source);
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    /*- Reading the measured value from the instrument ----------------------*/
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;
    
    /*- Checking for all kinds of errors reported thru the response string --*/
    Scan (comBuffer, "Overflow");
    if (NumFmtdBytes () == 8) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, "failed");
    if (NumFmtdBytes () == 6) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " CH1");
    if (NumFmtdBytes () == 4) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " CH2");
    if (NumFmtdBytes () == 4) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " MULT");
    if (NumFmtdBytes () == 5) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " ADD");
    if (NumFmtdBytes () == 4) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " XY");
    if (NumFmtdBytes () == 3) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " T1#T2");
    if (NumFmtdBytes () == 6) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " V1#V2");
    if (NumFmtdBytes () == 4) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " no TRIG");
    if (NumFmtdBytes () == 8) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " no CH");
    if (NumFmtdBytes () == 6) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    Scan (comBuffer, " [-[");
    if (NumFmtdBytes () == 4) {
        return MXOX802X_ERROR_INSTR_MEAS_FAILED;
    }
    
    
    /*- Measurement was performed successfully, scaning the value -----------*/
    if (Scan (comBuffer, "%f", output) < 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Read Trace                                                      */
/* Purpose:  This function reads the trace data from the instrument.         */
/*           Before reading data the trace catalog is looked up to make sure */
/*           the selected trace is available.                                */
/*           The DINT format is used for data transfer in order to obtain    */
/*           scaling factor, time increment and number of samples. The actual*/
/*           samples are transferred as binary 8 bit values.                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_readTrace (ViSession instrSession, ViInt32 trace,
                             ViReal64 _VI_FAR traceData[], ViPInt32 numberOfSamples,
                             ViPReal64 t0, ViPReal64 dt)
{
    ViUInt32 retCnt = 0;
    ViInt32 byteCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];
    ViByte fmtStr[100];
    ViReal64 scale;
    ViInt32 offset;
    ViInt32 i;
	ViUInt16 interface;                         
    mxOX802x_instrRange instrPtr;
    

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViInt32Range (trace, 0, 13))
        return VI_ERROR_PARAMETER2;

	viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr);

    if(instrPtr -> model == 0) {
        if (trace == 2) 
            trace--;
        else if (trace > 3 && trace < 8) 
            trace=trace-2;
        else if (trace == 10 || trace == 11) 
            trace=trace-4;
        else if (trace != 0)
            return VI_ERROR_PARAMETER2;
    }
    
    /*- Determining whether the acquisition is running or not ---------------*/
    Fmt (comBuffer, "%s<:INIT:CONT?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;

    /*- If the acquisition is running, can not read the data ----------------*/
    Scan (comBuffer, "ON");
    if (NumFmtdBytes () == 2)
        return MXOX802X_ERROR_INSTR_READ_TRACE;

    /*- Acquisition is not running ------------------------------------------*/
    
    /*- Setting timeout to 20 seconds to make sure the data transfer never --*/
    /*- times out. Transfering 8192 points (e.g. CH1A) takes approx. 8 sec.--*/
    if ((mxOX802x_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 20000)) < 0)
        return mxOX802x_status;

    /*- Reading Trace Catalog -----------------------------------------------*/
    Fmt (comBuffer, "%s<:FORM:DATA INT;:FORM:DINT ON;:TRAC:CAT?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 100, &retCnt)) < 0)
        return mxOX802x_status;

    /*- Searching for the specified trace name in the catalog, if not found -*/
    /*- returns error. ------------------------------------------------------*/
    if(instrPtr -> model == 0){
        if (FindPattern (comBuffer, 0, -1, tracStr2[trace], 1, 0) < 0)
            return MXOX802X_ERROR_INSTR_TRACE_NOT_AVAIL;
    }
    else {
        if (FindPattern (comBuffer, 0, -1, tracStr[trace], 1, 0) < 0)
            return MXOX802X_ERROR_INSTR_TRACE_NOT_AVAIL;
    }
    
    /*- Command to query specified trace ------------------------------------*/
    viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
    if(interface==VI_INTF_ASRL)
        viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE);

    if(instrPtr -> model == 0)
        Fmt (comBuffer, "%s<:TRACE:DATA? %s\n", tracStr2[trace]);
    else
        Fmt (comBuffer, "%s<:TRACE:DATA? %s\n", tracStr[trace]);
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, traceBuffer, 150, &retCnt)) < 0)
        return mxOX802x_status;

    /*- Creating formatting string to scan delta t --------------------------*/
    if(instrPtr -> model == 0)
        Fmt (fmtStr, "(ADIF=%s ( STD(VERSION 1992.0) DIM=X( TYPE IMPL SCALE %%f SIZE", tracStr2[trace]);
    else
        Fmt (fmtStr, "(ADIF=%s ( STD(VERSION 1992.0) DIM=X( TYPE IMPL SCALE %%f SIZE", tracStr[trace]);
    /*- Scanning delta t ----------------------------------------------------*/
    if (Scan (traceBuffer, fmtStr, dt) != 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    byteCnt = NumFmtdBytes ();
    
    /*- Scanning number of samples, Y offset and Y scaling factor -----------*/
    if (Scan (traceBuffer, "%s[i*]>%d[b4]) DIM=Y( TYPE EXPL SCALE %f OFFSET %d[b4] SIZE 255) DATA( CURVE( VAL#4%s[w4d]", (int)byteCnt, numberOfSamples, &scale, &offset) != 4)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;  
    
    byteCnt += NumFmtdBytes ();
    
    if ((mxOX802x_status = viRead (instrSession, traceBuffer + 150, 5 -(150-byteCnt)+ (int)*numberOfSamples, &retCnt)) < 0)
        return mxOX802x_status;

    if(interface==VI_INTF_ASRL) 
        viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR);
  
    /*- Scanning the trace data ---------------------------------------------*/
    if (Scan (traceBuffer, "%*i[i*b1uz]>%*f", (int)*numberOfSamples, (int)byteCnt, (int)*numberOfSamples, traceData) != 1) 
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    /*- Scaling scanned trace data ------------------------------------------*/
    for (i=0; i<*numberOfSamples; i++)
        traceData[i] = scale * (traceData[i] - offset);
    
    /* t0 is always zero ----------------------------------------------------*/
    *t0 = 0.0;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/

/*****************************************************************************/
/*-------- INSERT USER-CALLABLE INSTRUMENT-DEPENDENT ROUTINES HERE ----------*/
/*****************************************************************************/
           
/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
    ViStatus mxOX802x_status = VI_SUCCESS;
    
    if ((mxOX802x_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
        return mxOX802x_status;

    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_readInstrData (ViSession instrSession, ViInt16 numberBytesToRead,
                    ViChar _VI_FAR readBuffer[], ViPInt32 numBytesRead)
{
    ViStatus mxOX802x_status = VI_SUCCESS;
    *numBytesRead = 0L;
        
    if ((mxOX802x_status = viRead (instrSession, readBuffer, numberBytesToRead, numBytesRead)) < 0)
        return mxOX802x_status;

    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_reset (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;

    /*  Initialize the instrument to a known state.  */
    if ((mxOX802x_status = viWrite (instrSession, "*RST\n", 5, &retCnt)) < 0)
        return mxOX802x_status;

    if ((mxOX802x_status = mxOX802x_defaultInstrSetup (instrSession)) < 0)  
        return mxOX802x_status;
       
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_selfTest (ViSession instrSession, ViPInt16 testResult,
                    ViChar _VI_FAR testMessage[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;

    if ((mxOX802x_status = viWrite (instrSession, "*TST?\n", 6, &retCnt)) < 0)
        return mxOX802x_status;

    if ((mxOX802x_status = viScanf (instrSession, "%hd", testResult)) < 0)
        return mxOX802x_status;
        
    if (*testResult) {
        Fmt (testMessage, "Failed");
    }   
    else {
        Fmt (testMessage, "Passed");
    }   
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_errorQuery (ViSession instrSession, ViPInt32 errorCode,
                    ViChar _VI_FAR errorMessage[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViInt16 i;
    static mxOX802x_tStringValPair instrErrDescArray[] = {
        {0,     "Success"},
        {-100,  "Command error"},
        {-101,  "Invalid character"},
        {-102,  "Syntax error"},
        {-103,  "Invalid separator"},
        {-104,  "Data type error"},
        {-105,  "GET not allowed"},
        {-108,  "Parameter not allowed"},
        {-109,  "Missing parameter"},
        {-111,  "Header separator error"},
        {-112,  "Program mnemonic too long"},
        {-113,  "Undefined header"},
        {-114,  "Header suffix out of range"},
        {-121,  "Invalid character in number"},
        {-123,  "Numeric overflow"},
        {-124,  "Too many digits"},
        {-128,  "Numeric data not allowed"},
        {-130,  "Suffix error"},
        {-131,  "Invalid suffix"},
        {-138,  "Suffix not allowed"},
        {-140,  "Character data error"},
        {-141,  "Invalid character data"},
        {-144,  "Character data too long"},
        {-148,  "Character data not allowed"},
        {-150,  "String data error"},
        {-151,  "Invalid string data"},
        {-158,  "String data not allowed"},
        {-160,  "Block data error"},
        {-161,  "Invalid block data"},
        {-168,  "Block data not allowed"},
        {-170,  "Expression error"},
        {-171,  "Invalid expression"},
        {-178,  "Expression data not allowed"},
        {-200,  "Execution error"},
        {-211,  "Trigger ingnored"},
        {-221,  "Settings conflict"},
        {-222,  "Data out of range"},
        {-223,  "Too much data"},
        {-300,  "Device specific error"},
        {-310,  "System error"},
        {-350,  "Too many errors"},
        {-400,  "Query error"},
        {-410,  "Query INTERRUPTED"},
        {-420,  "Query UNTERMINATED"},
        {-430,  "Query DEADLOCKED"},
        {-440,  "Query UNTERMINATED after indefinite response"},
        {VI_NULL, VI_NULL}
    };
    
    if ((mxOX802x_status = viWrite (instrSession, ":SYST:ERR?\n", 11, &retCnt)) < 0)
        return mxOX802x_status;

    if ((mxOX802x_status = viScanf (instrSession, "%ld", errorCode)) < 0)
        return mxOX802x_status;

    for (i=0; instrErrDescArray[i].stringName; i++) {
        if (instrErrDescArray[i].stringVal == *errorCode) {
            Fmt (errorMessage, "%s<%s", instrErrDescArray[i].stringName);
            return (VI_SUCCESS);
        }
    }
    
    Fmt (errorMessage, "%s<Unknown Error %d[b4]", *errorCode);
    return (VI_WARN_UNKNOWN_STATUS);
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_errorMessage (ViSession instrSession, ViStatus statusCode,
                    ViChar _VI_FAR message[])
{
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViInt16 i;
    static mxOX802x_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,     "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,        "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,    "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,  "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,    "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
        {VI_ERROR_INSTR_PARAMETER9 ,  "ERROR: Parameter 9 out of range"},
        {VI_ERROR_INSTR_PARAMETER10,  "ERROR: Parameter 10 out of range"},
        {VI_ERROR_INSTR_PARAMETER11,  "ERROR: Parameter 11 out of range"},
        {VI_ERROR_INSTR_PARAMETER12,  "ERROR: Parameter 12 out of range"},
        {VI_ERROR_INSTR_PARAMETER13,  "ERROR: Parameter 13 out of range"},
        {VI_ERROR_INSTR_PARAMETER14,  "ERROR: Parameter 14 out of range"},
        {VI_ERROR_INSTR_PARAMETER15,  "ERROR: Parameter 15 out of range"},
        {MXOX802X_ERROR_INSTRUMENT_ERROR, "ERROR: Instrument specific error"},
        {MXOX802X_ERROR_INSTR_TRACE_NOT_AVAIL, "ERROR: Selected trace is not available"},
        {MXOX802X_ERROR_INSTR_MEAS_FAILED, "ERROR: Measurement failed, range overflow or no channel"},
        {MXOX802X_ERROR_INSTR_SET_EXPANSION, "ERROR: Error setting expansion when acquisition is running"},
        {MXOX802X_ERROR_INSTR_READ_TRACE, "ERROR: Attempt to read trace when acquisition is running"},
        {MXOX802X_ERROR_INSTR_SAVE_TRACE, "ERROR: Attempt to save trace when acquisition is running"},
        {VI_NULL, VI_NULL}
    };

    mxOX802x_status = viStatusDesc (instrSession, statusCode, message);
    if (mxOX802x_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                Fmt (message, "%s<%s", statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        Fmt (message, "%s<Unknown Error 0x%x[uw8p0]", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    mxOX802x_status = VI_SUCCESS;
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_revisionQuery (ViSession instrSession,
                    ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
   
    if ((mxOX802x_status = viWrite (instrSession, "*IDN?\n", 6, &retCnt)) < 0)
        return mxOX802x_status;

    if ((mxOX802x_status = viScanf (instrSession, "%*[^,],%*[^,],FV %[^\n]", instrRev)) < 0)
        return mxOX802x_status;
    
    Fmt (driverRev, "%s<%s", mxOX802x_REVISION);
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Front Panel Lockout                                             */
/* Purpose:  This function controls the instrument's front panel             */
/*           lockout state.                                                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_frontPanLock (ViSession instrSession, ViBoolean mode)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[BUFFER_SIZE];           

    /*- Check input parameter ranges ----------------------------------------*/
    if (mxOX802x_invalidViBooleanRange (mode))
        return VI_ERROR_PARAMETER2;

    /*- Command to lock/unlock the instrument's front panel -----------------*/
    Fmt (comBuffer, "%s<:SYST:KLOC %s\n", swOnOff[mode]);
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
    
    if ((mxOX802x_status = mxOX802x_instrStatus (instrSession)) < 0)        
        return mxOX802x_status;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX802x_close (ViSession instrSession)
{
    ViSession rmSession;
    ViStatus mxOX802x_status = VI_SUCCESS;
    mxOX802x_instrRange instrPtr;

    if ((mxOX802x_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
        return mxOX802x_status;
    if ((mxOX802x_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
        return mxOX802x_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    mxOX802x_status = viClose (instrSession);
    viClose (rmSession);

    return mxOX802x_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean mxOX802x_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean mxOX802x_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean mxOX802x_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the mxOX802x_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from mxOX802x_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus mxOX802x_initCleanUp (ViSession openRMSession,
                    ViPSession openInstrSession, ViStatus currentStatus)
{
    mxOX802x_instrRange instrPtr;

    if (viGetAttribute (*openInstrSession, VI_ATTR_USER_DATA, &instrPtr) >= 0)
	    if (instrPtr != NULL) 
    	    free (instrPtr);
    
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus mxOX802x_instrStatus (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViByte comBuffer[100];
    ViByte status;

    /*- Command to query Event Status Register ------------------------------*/
    Fmt (comBuffer, "%s<*ESR?\n");
    if ((mxOX802x_status = viWrite (instrSession, comBuffer, NumFmtdBytes (), &retCnt)) < 0) 
        return mxOX802x_status;
        
    if ((mxOX802x_status = viRead (instrSession, comBuffer, 50, &retCnt)) < 0)
        return mxOX802x_status;

    /*- Scanning the response -----------------------------------------------*/
    if (Scan (comBuffer, "%d[b1]", &status) != 1)
        return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
    /*- Testing bits 4 and 5 ------------------------------------------------*/
    if (status & 0x30)
        return MXOX802X_ERROR_INSTRUMENT_ERROR;
    
    return mxOX802x_status;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the mxOX802x_reset operation and by the   */
/*           mxOX802x_init function if the reset option has not been         */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus mxOX802x_defaultInstrSetup (ViSession instrSession)
{
    ViStatus mxOX802x_status = VI_SUCCESS;
    ViUInt32 retCnt = 0;
    ViByte rdBuffer[BUFFER_SIZE];
    mxOX802x_instrRange instrPtr;
    
    /* Determine if the structure has been initialized for the current VISA  */
    /* Session and malloc if it has not.                                     */
    if (mxOX802x_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
        return mxOX802x_status;
    
    if (instrPtr == NULL) 
        instrPtr = malloc (sizeof (struct mxOX802x_statusDataRanges));

    if ((mxOX802x_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, 
    						(ViUInt32)instrPtr)) < 0)
        return mxOX802x_status;                                       

	viWrite (instrSession, "\n*IDN?\n", 7, &retCnt);
	viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt);

	if(FindPattern (rdBuffer, 0, -1, "OX8027", 0, 0) != -1)
		instrPtr -> model = 1;
    else
		instrPtr -> model = 0;

    if ((mxOX802x_status = viWrite (instrSession, ":FORM:DATA INT;:FORM:DINT ON\n", 29, &retCnt)) < 0)
        return mxOX802x_status;

    return mxOX802x_status;
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
