#include <analysis.h>
#include "gpib.h"
#include "formatio.h"
#include "rs232.h"
#include "ox8020.h"

/*= Metrix OX8020 & OX8027 ================================================*/
/* LabWindows Instrument Driver CVI 3.1                                    */
/* Original Release:  Feb 14/95                                            */
/* By:  Francis VALOTEAU                                                   */
/* Originally written in C                                                 */
/* Modification History: None                                              */
/*                      07/05/95 : check_cat() becomes ox8020_check_cat()  */
/*                                 StrNCmp()   becomes ox8020_StrNCmp()    */
/*                                 read_param()  is deleted                */
/*                                 ox8020_size is added                    */
/*                                 globals instrument,version and indice   */
/*                                 become ox8020_instrument,ox8020_version */
/*								   ox8020_indice						   */
/*								   bd become LOCAL (static int bd)		   */
/*                                 ox8020_size is added                    */
/*                                 (updated after ox8020_wave_read)		   */
/*                                 Constants are deleted 				   */
/*                      08/02/96   if (interface) FlushInq in wave_read    */
/*=========================================================================*/

/*= GLOBALS ===============================================================*/
double  ox8020_xscale;          /* Time interval between two samples       */
double  ox8020_yscale;          /* Voltage interval between two quantums   */
double  ox8020_yoffset;         /* Position of OV (GND)                    */
int     ox8020_size;
int     ox8020_npoint;          /* Number of samples in the wave           */
int     ox8020_err;             /* Error number                            */
int     ox8020_port;            /* Number of serial port                   */
int     ox8020_instrument;      /* N of instrument (8020 or 8027)         */
int     ox8020_version;         /* Software Version of instrument          */
int     ox8020_indice;          /* Software indice of version              */
char    ox8020_state[60];       /* Response to instrument queries          */
char    ox8020_mesure[20];      /* Response to automatic measurements      */
char    ox8020_header[200];     /* Header of waveforms                     */

/*= INTERNAL DATA =========================================================*/
static int     bd;              /* Board Descriptor                        */
static int     interface;       /* 1 for RS232 , 0 for IEEE                */
static int     flag_klock;      /* KeyBoard Locked       (TRUE if 1)       */
static int     flag_anal_num;   /* Analog/Digital mode                     */
static int     flag_mode_acq;   /* Acquisition Mode                        */
static int     flag_ch1AHL;     /* Flag PEAK wave H or L (TRUE if 1)       */
static int     flag_ch1BHL;     /* Flag PEAK wave H or L (TRUE if 1)       */
static int     flag_ch2AHL;     /* Flag PEAK wave H or L (TRUE if 1)       */
static int     flag_ch2BHL;     /* Flag PEAK wave H or L (TRUE if 1)       */
static int     flag_visu_s2;    /* Flag Signal 2 dispayed (TRUE if 1)      */
static int     time_base;       /* Time Base                               */
static int     flag_acq_run;    /* Flag acquisition (Running if 0)         */
static short   setup[10][57];   /* Instrument setups                       */

/*= UTILITY ROUTINES ======================================================*/
int     ox8020_device_closed (void);
int     ox8020_StrNCmp (char *, char *, int);
int     ox8020_check_cat(void);


/* ------------------------------------------------- */
/* Routine name :   init                             */
/* Function     :   Initialize instrument            */
/* ------------------------------------------------- */
int ox8020_init (int inter,int reset,int klock,int i3e_adr,int rs_port,
                 int baud_rate,int parity)
{
    char idn[50];
    char echo[40];
    int n;
    char cmd[200];
    int i__ret,i;
    char *timebase[29];

    /* Timebase range table initialization */
    timebase[0] = "0.1E-6\n";
    timebase[1] = "0.2E-6\n";
    timebase[2] = "0.5E-6\n";
    timebase[3] = "1E-6\n";
    timebase[4] = "2E-6\n";
    timebase[5] = "5E-6\n";
    timebase[6] = "10E-6\n";
    timebase[7] = "20E-6\n";
    timebase[8] = "50E-6\n";
    timebase[9] = "0.1E-3\n";
    timebase[10] = "0.2E-3\n";
    timebase[11] = "0.5E-3\n";
    timebase[12] = "1E-3\n";
    timebase[13] = "2E-3\n";
    timebase[14] = "5E-3\n";
    timebase[15] = "10E-3\n";
    timebase[16] = "20E-3\n";
    timebase[17] = "50E-3\n";
    timebase[18] = "100E-3\n";
    timebase[19] = "200E-3\n";
    timebase[20] = "500E-3\n";
    timebase[21] = "1\n";
    timebase[22] = "2\n";
    timebase[23] = "5\n";
    timebase[24] = "10\n";
    timebase[25] = "20\n";
    timebase[26] = "50\n";
    timebase[27] = "100\n";
    timebase[28] = "200\n";

    ox8020_err = 0;
    flag_klock=klock;
    bd=0;

    /* Check range of parameters */
    if ((inter<0) || (inter > 1))  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if ((reset<0) || (reset > 1))  {
        ox8020_err =  -2;
        return ox8020_err;
    }
    if ((klock<0) || (klock > 1))  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (( -(i3e_adr < 0) |  -(i3e_adr > 30)) != 0)  {
        ox8020_err =  -4;
        return ox8020_err;
    }
    if (( -(rs_port < 1) |  -(rs_port > 8)) != 0)  {
        ox8020_err = -5;
        return ox8020_err;
    }
    if ((baud_rate<110) || (baud_rate >9600))  {
        ox8020_err =  -6;
        return ox8020_err;
    }
    if ((parity<0) || (parity > 2))  {
        ox8020_err =  -7;
        return ox8020_err;
    }

    if (!inter)
        {
        /* ******** IEEE ******** */
        /* if device not opened */
        if (bd <= 0)
            {
            CloseInstrDevs("ox8020");
            /* Open Device */
            bd = OpenDev ("", "ox8020");
            if (bd <= 0)
                { /* Error : Unable to open device */
                ox8020_err = 220;
                return ox8020_err;
                }
            }
        /*  Change the primary address of the device    */
        if (ibpad (bd, i3e_adr) < 0)
            {
            ox8020_err = 233;
            return ox8020_err;
            }
        /* Set Iime-out to 10 S */
        ibtmo (bd, 13);
        /* Set EOS = 'LF' */
        ibeos (bd, 0x1C0A);
        }
    else
        {  /* ******** RS232 ******** */
        bd = 1;
        OpenComConfig (rs_port, "", baud_rate,parity, 8,1, 10000, 512);
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
        SetComTime (rs_port, 10.0);   /* Set timeout on RS232 interface */
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
   	    SetXMode (rs_port, 1);       /* protocole XON/XOFF on */
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
        FlushInQ (rs_port);
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
        FlushOutQ (rs_port);
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
        ox8020_port = rs_port;
        }

    interface = inter;

    /* identify instrument and software version */
    if (ox8020_write_data("*IDN?\n") != 0)
        return ox8020_err;
    if (ox8020_read_data(echo,40) != 0)
        return ox8020_err;
    i__ret = Scan (echo,"%s>%s[dt#]%d%s[dt#]%d%s[dt#]%d",&ox8020_instrument,&ox8020_version,&ox8020_indice);
    if (ox8020_instrument!=8020  && ox8020_instrument!=8027)
        {
        ox8020_err = -8; /* unknowed instrument identified */
        return ox8020_err;
        }

    /* If enabled : instrument RESET */
    if (reset==1)
        {
        i__ret = ox8020_write_data ("*RST\n");
        flag_anal_num = 0;
        flag_mode_acq = 1;
        time_base= 8;    /* 50s/div */
        }
    else
        {
        /* Read "ACQUIRE:MODE" ":MODE" "TIMEbase:RANGe?"             */
        /* then store in flag_mode_acq , flag_anal_num , time_base */

        /* Read ACQUIRE:MODE */
        if (ox8020_instrument_query(2,1)!=0)
            return ox8020_err;
        if ((ox8020_StrNCmp(ox8020_state,"SINGLE",6)==0))
            flag_mode_acq=0;
        else
            if ((ox8020_StrNCmp(ox8020_state,"REFRESH",7)==0))
                flag_mode_acq=1;
            else
                flag_mode_acq=2;

        /* Read :MODE */
        if (ox8020_instrument_query(0,1)!=0)
            return ox8020_err;
        if ((ox8020_StrNCmp(ox8020_state,"NUM",3)==0))
            flag_anal_num=1;
        else
            flag_anal_num=0;

        /* Read Timebase */
        if (ox8020_instrument_query(25,1)!=0)
            return ox8020_err;
        i__ret=FindPattern (ox8020_state, 0, -1, "\n", 0, 0);
        if (i__ret==-1)
            {
            i=StringLength (ox8020_state);
            ox8020_state[i]=0x0A;
            ox8020_state[i+1]=0;
            }
        i=0;
        while ((i<29)&&(ox8020_StrNCmp(ox8020_state, timebase[i],StringLength(ox8020_state))!=0))
            i++;
        time_base=i;
        }

    /* Lock or Unlock Front Panel */
    if (klock==1)
       i__ret = ox8020_write_data ("SYST:KLOCK ON\n");
    else
        i__ret = ox8020_write_data ("SYST:KLOCK OFF\n");

    flag_acq_run = 0;

    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   setup                            */
/* Function     :   Save/Recall Instrument setup     */
/* ------------------------------------------------- */
int ox8020_setup (int function , int storage , int number , char *filename)
{
int i_ret , file_id , i;
char commande[57];

    /* **************** Parameter validation ************* */
    if ((function != 0) && (function != 1)) {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if ((storage != 0) && (storage != 1)) {
        ox8020_err =  -2;
        return ox8020_err;
    }
    if ((number<1)||(number > 10))  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    /* *************************************************** */
    ox8020_err = 0;

    if (ox8020_device_closed () != 0)
        return ox8020_err;


    if (function==0)
        {
        /* Disable  XON/XOFF (Binary Format) */
        if (interface)
        	SetXMode (ox8020_port, 0);

        /* Set no EOS (end with EOI) */
        if (!interface) ibeos (bd, 0x0);

        /* Save Setup */
        ox8020_write_data ("*LRN?\n");


        if (!ox8020_err)
            {
            if (storage==0)
                /* Store Setup into RAM */
                {
                i_ret=ox8020_read_data (commande,57 );
                if (i_ret!=0) return (i_ret);
                for (i=0;i<57;i++)
                    setup[number-1][i]=0x00FF&commande[i];
                }
            else
                {
                /* Store Setup into file */
                ox8020_read_data (commande,57 );
                /* Open File :  Write only, append, ASCII */
                file_id = OpenFile (filename, 2, 0, 1);
                if (file_id ==  -1)
                    {
                    /* error */
                    ox8020_err =  -4;
                    return ox8020_err;
                    }
                WriteFile (file_id, commande, 57);
                CloseFile (file_id);
                }
            }
        }
    else
        {
        /* Recall Setup */
        if (!ox8020_err)
            {
            if (storage==0)
                {
                /* Recall from RAM */
                if (setup[number-1][0] != '#')
                    {
                    ox8020_err=-5;
                    }
                else
                    {
                     /* Send Command SYST:SET */
                    ox8020_write_data ("SYST:SET ");
                    for (i=0;i<57;i++)
                        commande[i]=0x00FF & setup[number-1][i];
                    /* Send Instrument Setup (Binary Code) */
                    if (!interface)  /* ******** IEEE ******** */
                        {
                        i_ret=ibwrt(bd, commande, 57);
                        }
                    else            /* ******** RS232 ******** */
                        {
                        i_ret=ComWrt (ox8020_port,commande , 57 );
                        /* Wait for the output buffer empty */
                        while (GetOutQLen (ox8020_port)!=0) {}
                        }
                    }
                }
            else
                {
                /* Recall from File */
                /* Open File :  Read only, truncate, ASCII */
                file_id = OpenFile (filename, 1, 0, 1);
                if (file_id ==  -1)
                    {
                    /* error */
                    ox8020_err =  -4;
                    return ox8020_err;
                    }
                /* Send Command SYST:SET */
                ox8020_write_data ("SYST:SET ");

                if (!interface)
                    {
                    ReadFile (file_id,commande,57 );
                    i_ret=ibwrt(bd, commande, 57);
                    }
                else
                    {
                    ComFromFile (ox8020_port, file_id, 57, -1);
                    /* Wait for the output buffer empty */
                    while (GetOutQLen (ox8020_port)!=0) {}
                    }
                CloseFile (file_id);
                }
            }
        }
    /* Enable  XON/XOFF  */
    if (interface)
    	SetXMode (ox8020_port, 1);

    /* Set EOS = 'LF' */
    if (!interface) ibeos (bd, 0x1C0A);

    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   cont_inst                        */
/* Function     :   MODE & AUTOSET                   */
/* ------------------------------------------------- */

int ox8020_cont_inst (int mode_an_num, int autoscale)
{

    /* **************** Parameter validation ************* */
    if (( -(mode_an_num != 0) &  -(mode_an_num != 1)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (( -(autoscale != 0) &  -(autoscale != 1)) != 0)  {
        ox8020_err =  -2;
        return ox8020_err;
    }
    /* *************************************************** */
    ox8020_err = 0;

    if (ox8020_device_closed () != 0)
        return ox8020_err;

    if (!mode_an_num)
        ox8020_write_data ("MODE AN\n");
    else
        ox8020_write_data ("MODE NUM\n");

    if (autoscale == 1)
        ox8020_write_data ("AUT\n");

    flag_anal_num = mode_an_num;
    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :    channel                         */
/* Function     :    Channels settings               */
/* ------------------------------------------------- */
int ox8020_channel (int channel_num, int coupling, int range , int invert_ch2)
{
    int n;
    char cmd[200];
    char *sensib[14 - 1 + 1];
    int i__ret;

    ox8020_err = 0;

    /* sensibility table initialization */
    sensib[0] = "1E-3";
    sensib[1] = "2E-3";
    sensib[2] = "5E-3";
    sensib[3] = "10E-3";
    sensib[4] = "20E-3";
    sensib[5] = "50E-3";
    sensib[6] = "100E-3";
    sensib[7] = "200E-3";
    sensib[8] = "500E-3";
    sensib[9] = "1";
    sensib[10] = "2";
    sensib[11] = "5";
    sensib[12] = "10";
    sensib[13] = "20";

    /* **************** Parameter validation ************* */
    if (( -(channel_num != 1) &  -(channel_num != 2)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (~( -(coupling >= 0) &  -(coupling <= 2 )) != 0)  {
        ox8020_err =  -2;
        return ox8020_err;
    }
    if (~( -(range >= 0) &  -(range < 14)) != 0)  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (( -(invert_ch2 != 1) &  -(invert_ch2 != 0)) != 0)  {
        ox8020_err =  -4;
        return ox8020_err;
    }
    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    FillBytes (cmd, 0, 200, 0);
    /* coupling configuration */
    switch (coupling)  {
    case 0:
        n = Fmt (cmd, "%s<INP%d:COUP AC\n", channel_num);
        i__ret = ox8020_write_data (cmd);
        break;
    case 1:
        n = Fmt (cmd, "%s<INP%d:COUP DC\n", channel_num);
        i__ret = ox8020_write_data (cmd);
        break;
    case 2:
        n = Fmt (cmd, "%s<INP%d:COUP GND\n", channel_num);
        i__ret = ox8020_write_data (cmd);
        break;
    default:
        break;
    }

    /* Sensibility configuration */
    if (!ox8020_err)  {
        FillBytes (cmd, 0, 200, 0);
        n = Fmt (cmd, "%s<INP%d:RANG %s\n", channel_num, sensib[range]);
        i__ret = ox8020_write_data (cmd);
    }
    /* channel 2 invert state configuration */
    if (!invert_ch2)
        ox8020_write_data ("INP2:INV OFF\n");
    else
        ox8020_write_data ("INP2:INV ON\n");
    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   channel save                     */
/* Function     :   Configure the channels save state*/
/* ------------------------------------------------- */
int ox8020_channel_save (int l__channel, int state)
{
    int n;
    char cmd[200];
    int channel;
    int i__ret;

    /*  - Modified value parameter */
    channel = l__channel;
    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if (~( -(channel >= 1) &  -(channel <= 4)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (( -(ox8020_instrument == 8020) &  -(channel == 2)) != 0)
        channel = 1;
    if (( -(ox8020_instrument == 8020) & ( -(channel == 4) |
                        -(channel == 3))) != 0)
        channel = 2;
    if (( -(state != 1) &  -(state != 0)) != 0)  {
        ox8020_err =  -2;
        return ox8020_err;
    }
    if (flag_acq_run==1) {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    FillBytes (cmd, 0, 200, 0);
    /* channel state configuration */
    if (!state)  {
        n = Fmt (cmd, "%s<DISP:SAVE%d OFF\n", channel);
        i__ret = ox8020_write_data (cmd);
    }
    else  {
        n = Fmt (cmd, "%s<DISP:SAVE%d ON\n", channel);
        i__ret = ox8020_write_data (cmd);
    }
    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   Timebase                         */
/* Function     :   Configure time base settings     */
/* ------------------------------------------------- */
int ox8020_timebase (int expend,int mode_auto,int delay1,int range)
{
    int n;
    char cmd[200];
    char *timebase[29];
    int i__ret;

    ox8020_err = 0;

    /* Timebase range table initialization */
    timebase[0] = "0.1E-6";
    timebase[1] = "0.2E-6";
    timebase[2] = "0.5E-6";
    timebase[3] = "1E-6";
    timebase[4] = "2E-6";
    timebase[5] = "5E-6";
    timebase[6] = "10E-6";
    timebase[7] = "20E-6";
    timebase[8] = "50E-6";
    timebase[9] = "0.1E-3";
    timebase[10] = "0.2E-3";
    timebase[11] = "0.5E-3";
    timebase[12] = "1E-3";
    timebase[13] = "2E-3";
    timebase[14] = "5E-3";
    timebase[15] = "10E-3";
    timebase[16] = "20E-3";
    timebase[17] = "50E-3";
    timebase[18] = "100E-3";
    timebase[19] = "200E-3";
    timebase[20] = "500E-3";
    timebase[21] = "1";
    timebase[22] = "2";
    timebase[23] = "5";
    timebase[24] = "10";
    timebase[25] = "20";
    timebase[26] = "50";
    timebase[27] = "100";
    timebase[28] = "200";

    /* **************** Parameter validation ************* */
    if (( -(expend != 0) &  -(expend != 1)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if ((mode_auto!=0)&&(mode_auto!=1))  {
        ox8020_err=-2;
        return ox8020_err;
    }
    if (~( -(delay1 >= 0) &  -(delay1 <= 2)) != 0)  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if ((range <0) || (range > 28))  {
        ox8020_err =  -4;
        return ox8020_err;
    }
    /* Check Timebase range */

    if (flag_anal_num == 1)
        { /* Memory mode */
        if (flag_mode_acq == 2)
            { /* ROLL mode : Range = [20..28] (500E-3..200 s/div) */
            if ((range < 20)||(range >28))
                {
                ox8020_err =  -5;
                return ox8020_err;
                }
            }
        }
    else
        { /* Analogic mode : Range = [2..19] (200E-3..0.5E-3 s/div) */
        if ((range < 2)||(range > 19))
            {
            ox8020_err =  -6;
            return ox8020_err;
            }
        }
    if ((flag_anal_num!=0) && (delay1 == 1 ))  {
        ox8020_err =  -7;
        return ox8020_err;
    }


    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    /* expension state setting */
    if (!expend)
        ox8020_write_data ("TIM:EXPH OFF\n");
    else
        ox8020_write_data ("TIM:EXPH ON\n");

    /* AUTO mode setting */
    if (!mode_auto)
        ox8020_write_data ("TIM:MODE NORM\n");
    else
        ox8020_write_data ("TIM:MODE AUTO\n");

 /* Delay mode setting */
    if (delay1 == 0)
        ox8020_write_data ("TIM:DEL OFF\n");
    else  if (delay1 == 1)
        ox8020_write_data ("TIM:DEL SEAR\n");
    else  if (delay1 == 2)
        ox8020_write_data ("TIM:DEL DEL\n");

    /* Timebase range setting */
    FillBytes (cmd, 0, 200, 0);
    n = Fmt (cmd, "%s<TIM:RANGE %s\n", timebase[range]);
    i__ret = ox8020_write_data (cmd);

    time_base=range;
    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   Trigger                          */
/* Function     :   Configure Trigger settings       */
/* ------------------------------------------------- */
int ox8020_trigger (int ptp,int slope,int source,int coupling)
{

    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if (( -(ptp != 0) &  -(ptp != 1)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (( -(slope != 0) &  -(slope != 1)) != 0)  {
        ox8020_err =  -2;
        return ox8020_err;
    }
    if (~( -(source >= 0) &  -(source <= 4)) != 0)  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (~( -(coupling >= 0) &  -(coupling <=5)) != 0)  {
        ox8020_err =  -4;
        return ox8020_err;
    }
    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    /* peak to peak state setting */
    switch (ptp)  {
    case 0:
        ox8020_write_data (":TRIG:MODE NORM\n");
        break;
    case 1:
        ox8020_write_data (":TRIG:MODE PTP\n");
        break;
    default:
        break;
    }

    /* trigger slope setting */
    switch (slope)  {
    case 0:
        ox8020_write_data (":TRIG:SLOP +\n");
        break;
    case 1:
        ox8020_write_data (":TRIG:SLOP -\n");
        break;
    default:
        break;
    }

    /* trigger source setting */
    switch (source)  {
    case 0:
        ox8020_write_data (":TRIG:SOUR CH1\n");
        break;
    case 1:
        ox8020_write_data (":TRIG:SOUR CH2\n");
        break;
    case 2:
        ox8020_write_data (":TRIG:SOUR ALT\n");
        break;
    case 3:
        ox8020_write_data (":TRIG:SOUR LINE\n");
        break;
    case 4:
        ox8020_write_data (":TRIG:SOUR EXT\n");
        break;
    default:
        break;
    }

    /* trigger coupling setting */
    switch (coupling)  {
    case 0:
        ox8020_write_data (":TRIG:COUP DC\n");
        break;
    case 1:
        ox8020_write_data (":TRIG:COUP AC\n");
        break;
    case 2:
        ox8020_write_data (":TRIG:COUP LF\n");
        break;
    case 3:
        ox8020_write_data (":TRIG:COUP HF\n");
        break;
    case 4:
        ox8020_write_data (":TRIG:COUP TVH\n");
        break;
    case 5:
        ox8020_write_data (":TRIG:COUP TVV\n");
        break;
    default:
        break;
    }

    return ox8020_err;

}

/* ------------------------------------------------- */
/* Routine name :   Acquire                          */
/* Function     :   Configure the acquisition        */
/* ------------------------------------------------- */
int ox8020_acquire (int mode_acq,int pretrig,int control,
                    int envelop , int glitch , int sync_wait)
{
    int n;
    char cmd[200];
    int i__ret;

    ox8020_err = 0;

    /* **************** Parameter validation ************* */
    if (~( -(mode_acq >= 0) &  -(mode_acq <= 2)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (ox8020_instrument==8020)
        {
        if (~( -(pretrig >=  -1) &  -(pretrig < 5)) != 0)
            {
            ox8020_err =  -2;
            return ox8020_err;
            }
        }
    else
        {
        if (~( -(pretrig >=  -1) &  -(pretrig < 9)) != 0)
            {
            ox8020_err =  -2;
            return ox8020_err;
            }
        }
    if (( -(envelop != 0) &  -(envelop != 1)) != 0)  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (( -(control != 0) &  -(control != 1)) != 0)  {
        ox8020_err =  -4;
        return ox8020_err;
    }
    if ((sync_wait != 0) && (sync_wait != 1)) {
        ox8020_err =  -5;
        return ox8020_err;
    }
    if (( -(glitch != 0) &  -(glitch != 1)) != 0)  {
        ox8020_err =  -6;
        return ox8020_err;
    }
    if ((mode_acq==2)&&(time_base<20)) {
        ox8020_err =  -7;
        return ox8020_err;
    }

    if ((pretrig ==  -1) &  (mode_acq!= 2)) {
        ox8020_err =  -8;
        return ox8020_err;
    }
    if ((envelop == 1)&&(( mode_acq != 1)||(time_base>19))) {
        ox8020_err = -9;
        return ox8020_err;
    }
    if ((sync_wait==1)&&(mode_acq!=0)) {
        ox8020_err = -10;
        return ox8020_err;
    }
    if ((ox8020_instrument==8020)&&(glitch==1))
        {
        ox8020_err =  -11;
        return ox8020_err;
        }


    /* *************************************************** */

    if (ox8020_device_closed () != 0)
        return ox8020_err;

    /* Abort Acquisition if enabled */
    if (control !=1)
        {
        if (mode_acq == 1)    /* mode refresh */
            ox8020_write_data ("INIT:CONT OFF\n");
        else                        /* mode roll ou single */
            ox8020_write_data ("INIT:CONT OFF\n");
        }

    /* Acquisition mode setting */
    switch (mode_acq)  {
    case 0:
        ox8020_write_data ("ACQ:MODE SING\n");
        break;
    case 1:
        ox8020_write_data ("ACQ:MODE REFR\n");
        break;
    case 2:
        ox8020_write_data ("ACQ:MODE ROLL\n");
        break;
    default:
        break;
    }

    /* envelop mode setting */
    if (!envelop)
        ox8020_write_data ("ACQ:PEAK OFF\n");
    else
        ox8020_write_data ("ACQ:PEAK ON\n");

    /* pretrig value setting */
    n = Fmt (cmd, "%s<ACQ:REF %d\n", pretrig);
    i__ret = ox8020_write_data (cmd);

    /* Glitch Capture setting */
    if (!glitch)
        ox8020_write_data ("ACQ:GLITCH OFF\n");
    else
        ox8020_write_data ("ACQ:GLITCH ON\n");

    /* Acquisition start configuration */
    if (control == 1)
        {
        if (mode_acq == 1)    /* mode refresh */
            ox8020_write_data ("INIT:CONT ON\n");
        else                        /* mode roll ou single */
            {
            if (sync_wait==0)
                ox8020_write_data ("INIT\n");
            else
                ox8020_write_data ("INIT;*WAI\n");
            }
        }
    flag_mode_acq = mode_acq;
    flag_acq_run = control;
    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   Display                          */
/* Function     :   Configure display settings       */
/* ------------------------------------------------- */
int ox8020_display (mode1,indbank,hor_exp,hor_pos,interpol,filter)
int mode1;
int indbank;
int hor_exp;
int hor_pos;
int interpol;
int filter;
{
    int n;
    char cmd[200];
    int i__ret;
    char *bank[4];

    /* Bank table initialization */
    bank[0] = "A";
    bank[1] = "B";
    bank[2] = "AB";
    bank[3] = "BA";

    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if (( -(mode1 < 0) |  -(mode1 > 6)) != 0)
        {
        ox8020_err =  -1;
        return ox8020_err;
        }
    if ((indbank<0)||(indbank>3))
        {
        ox8020_err =  -2;
        return ox8020_err;
        }
    if (( -(hor_exp < 0) |  -(hor_exp > 7)) != 0)  {
        ox8020_err =  -3;
        return ox8020_err;
    }
    if (ox8020_instrument == 8020)
        {
        if (hor_pos > 4000)
            {
            ox8020_err =  -4;
            return ox8020_err;
            }
        if (hor_exp < 1)
            {
            ox8020_err =  -3;
            return ox8020_err;
            }
        }
    else
        {
        if (( -(hor_pos < 0) |  -(hor_pos > 8000)) != 0)
            {
            ox8020_err =  -4;
            return ox8020_err;
            }
        }
    if (( -(interpol < 0) |  -(interpol > 2)) != 0)  {
        ox8020_err =  -5;
        return ox8020_err;
    }
    if (( -(filter != 0) &  -(filter != 1)) != 0)  {
        ox8020_err =  -6;
        return ox8020_err;
    }
    if ((mode1== 6) && (!flag_anal_num))
        {
        ox8020_err =  -7;
        return ox8020_err;
        }
    if (((mode1==0)||(mode1==3)) && (indbank==3))
        {
        ox8020_err =  -8;
        return ox8020_err;
        }
    if ((hor_exp!=3)&&(flag_acq_run==1))
        {
        ox8020_err =  -9;
        return ox8020_err;
        }
    /* *************************************************** */

    if (ox8020_device_closed () != 0)
        return ox8020_err;

    /* Horizontal Expansion and interpolation  */
    switch (interpol)
        {
        case 0   : ox8020_write_data ("DISP:FILT OFF\n");break;
        case 1: ox8020_write_data ("DISP:FILT LN\n"); break;
        case 2: ox8020_write_data ("DISP:FILT OFF\n");break;
        }
    n = Fmt (cmd, "%s<DISP:TRAC:X:PDIV %d\n", hor_exp);
    i__ret = ox8020_write_data (cmd);

    /* display mode setting */
    switch (mode1)  {
    case 0:
        ox8020_write_data ("DISP:MODE CH1\n");
        break;
    case 1:
        ox8020_write_data ("DISP:MODE ALT\n");
        break;
    case 2:
        ox8020_write_data ("DISP:MODE CHOP\n");
        break;
    case 3:
        ox8020_write_data ("DISP:MODE CH2\n");
        break;
    case 4:
        ox8020_write_data ("DISP:MODE XY\n");
        break;
    case 5:
        ox8020_write_data ("DISP:MODE ADD\n");
        break;
    case 6:
        ox8020_write_data ("DISP:MODE MULT\n");
        break;
    default:
        break;
    }

    /* for ox8027 only : Bank switching */
    if (ox8020_instrument == 8027)
        {
        FillBytes (cmd, 0, 200, 0);
        n = Fmt (cmd, "%s<DISP:TRAC:FEED %s\n", bank[indbank]);
        i__ret = ox8020_write_data (cmd);
        }

    /* Window configuration */
    n = Fmt (cmd, "%s<DISP:TRAC:X:LEFT %d\n", hor_pos );
    i__ret = ox8020_write_data (cmd);

    /* Horizontal Expansion and interpolation (if Acquisition not running) */
    if (interpol==2)
        {
         n = Fmt (cmd, "%s<DISP:TRAC:X:PDIV %d\n", hor_exp);
            i__ret = ox8020_write_data (cmd);

            ox8020_write_data ("DISP:FILT SN\n");
        }
    /* filter state setting */
    if (!filter)
        ox8020_write_data ("OUTP:FILT OFF\n");
    else
        ox8020_write_data ("OUTP:FILT ON\n");


    return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   wave_read                        */
/* Function     :   Read Waveform and store in array */
/*                  and if enabled in a file         */
/* ------------------------------------------------- */
int ox8020_wave_read ( int waveform, int head, int peak ,
                    int file , double array[], char *filename)
{
    static unsigned char wave[8300];
    static unsigned char target[8300];


    int  i,j;
    int  itemp;
    int  i__ret;
    int  form;
    int  sample_counter;
    int  i_start;
    int  count;
    char chaine[50];
    char wave_name[10];
    char cmd[20];

    int npoint;
    int ic;
    int file_id;
    char tampon[20];

    ox8020_err = 0;
    ox8020_xscale=0;
    ox8020_yscale=0;
    ox8020_yoffset=0;
    form=3;

    if (interface)
    	FlushInQ (ox8020_port);
    count=0;
    if (waveform<4)
        if (ox8020_instrument==8020) Clear1D (array, 4096);
        else Clear1D (array, 8192);
    else Clear1D (array, 1024);
    FillBytes (ox8020_header, 0, 200, 0);


    if (ox8020_device_closed () != 0)
        return (0);


    /* Analyse the catalog (waves available) */
    if (ox8020_check_cat()!=0)
        return 0;

    /* **************** Parameter validation ************* */
    if ((waveform < 0) || (waveform > 5))
        {
        ox8020_err =  -1;
        return (0);
        }
    if (( -(head != 0) &  -(head != 1)) != 0)
        {
        ox8020_err =  -2;
        return (0);
        }
    if ((peak < 0) || (peak > 1))
        {
        ox8020_err =  -3;
        return (0);
        }
    if ((file < 0) || (file > 1))
        {
        ox8020_err =  -4;
        return (0);
        }
    if ((peak==1)&&(waveform>3))
        {
        ox8020_err = -5;
        return (0);
        }
    if ((waveform==5)&&(flag_visu_s2==0))
        {
        ox8020_err = -6;
        return (0);
        }

    /* *************************************************** */


    /* Set BINARY Data Format */
    /* Disable  XON/XOFF */
    if (interface)
   		SetXMode (ox8020_port, 0);
    if (ox8020_write_data ("FORM INT\n")!=0)
        return (0);

    /* Set ADIF header */
    if (ox8020_write_data ("FORM:DINT ON\n")!=0)
        return (0);

    /* Calculate Samples number */
    if (waveform > 3)
       ox8020_npoint = 1024;              /* S1,S2             : 1024  */
    else
        if (ox8020_instrument == 8020)
            ox8020_npoint = 4096;      /* CH1,CH2 (ox8020)    : 4096 */
        else
            ox8020_npoint = 8192;       /*CH1A,CH1B,CH2A,CH2B : 8192 */

    /* Set Source Channel */
    if (ox8020_instrument == 8020)
        {
        switch (waveform)
            {
            case 0:
            case 1:
                i__ret = Fmt (wave_name, "%s<%s", "CH1");
                break;
            case 2:
            case 3:
                i__ret = Fmt (wave_name, "%s<%s", "CH2");
                break;
            case 4:
                i__ret = Fmt (wave_name, "%s<%s", "S1");
                break;
            case 5:
                i__ret = Fmt (wave_name, "%s<%s", "S2");
                break;
            default:
                break;
            }
        }
    else
        {
        switch (waveform)
            {
            case 0:
                i__ret = Fmt (wave_name, "%s<%s", "CH1A");
                break;
            case 1:
                i__ret = Fmt (wave_name, "%s<%s", "CH1B");
                break;
            case 2:
                i__ret = Fmt (wave_name, "%s<%s", "CH2A");
                break;
            case 3:
                i__ret = Fmt (wave_name, "%s<%s", "CH2B");
                break;
            case 4:
                i__ret = Fmt (wave_name, "%s<%s", "S1");
                break;
            case 5:
                i__ret = Fmt (wave_name, "%s<%s", "S2");
                break;
            default:
                break;
            }
        }

    if (peak==0)
        /* Read CH1|CH2|S1|S2 (ox8020) */
        /* or   CH1A|CH1B|CH2A|CH2B|S1|S2 (ox8027) */
        i__ret = Fmt (cmd, "%s<TRAC:DATA? %s\n", wave_name);
    else
        /* Read CH1L|CH1H|CH2L|CH2H (ox8020) */
        /* or   CH1AL|CH1AH|CH1BL|CH1BH|CH2AL|CH2AH|CH2BL|CH2BH (ox8027) */
        {
        if (ox8020_instrument==8020)
            {
            switch (waveform)
                {
                case 0:
                case 1:
                    if (flag_ch1AHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                case 2:
                case 3:
                    if (flag_ch2AHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                }
            }
        else /* OX8027 */
            {
            switch (waveform)
                {
                case 0:
                    if (flag_ch1AHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                case 1:
                    if (flag_ch1BHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                case 2:
                    if (flag_ch2AHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                case 3:
                    if (flag_ch2BHL==1)
                        {
                        i__ret = Fmt (cmd, "%s<TRAC:DATA? %sH\n", wave_name);
                        }
                    else
                        {
                        ox8020_err=-7;
                        return (0);
                        }
                    break;
                }
            }
        }

    /* Send "TRACE? <wave>" */
    if (ox8020_write_data(cmd)!=0)
        return (0);

    /* ******  Read Header ADIF ********* */
    if (!interface)
        {
        /* Set EOS = '#' */
        ibeos (bd, 0x1C23);
        /* Read header */
        ibrd (bd, ox8020_header, 200);
        }
    else
        {
        /* Bug with LW CVI 3.0.1                             */
        /* ComRdTerm (ox8020_port, ox8020_header, 200, '#'); */
        /* Replace with :                                    */
        i=0;
        ox8020_header[i]=ComRdByte (ox8020_port);
        if (rs232err!=0)
            {
            ox8020_err=231;
            return ox8020_err;
            }
        while ((ox8020_header[i]!='#')&&(i<199)&&(rs232err==0))
            {
            i++;
            ox8020_header[i]=ComRdByte (ox8020_port);
            }
        /* In use with ComRdTerm only                           */
        /* i__ret=StringLength (ox8020_header);                 */
        /* ox8020_header[i__ret]='#';                           */
        }

    /* ****** Read Waveform ************* */
    if (waveform<4)
        if (ox8020_instrument==8020)
             count=5+4096+4+1;  /* '44096'+ Data + ')))) ' */
        else count=5+8192+4+1;  /* '48192'+ Data + ')))) ' */
    else     count=5+1024+4+1;  /* '41024'+ Data + ')))) ' */

    if (!interface)
         {
         if ((ox8020_instrument==8027)&&(ox8020_version==1)&&(ox8020_indice==3))
            {
            /* Set EOS = ' ' */
            ibeos (bd, 0x1C20);
            sample_counter=0;
            FillBytes (target, 0, 8300, 0);
            while (sample_counter<count)
                {
                i__ret=ibrd (bd, wave, 8300);
                if ((i__ret&0x8000)!=0)
                    {
                    ox8020_err=iberr;
                    return sample_counter;
                    }
                CopyString (target, sample_counter, wave, 0, ibcnt);
                sample_counter+=ibcnt;
                }
            }
         else
            {
            /* Set no EOS (end with EOI) */
            ibeos (bd, 0x0);
            i__ret=ibrd (bd, target , 8300);
            sample_counter=ibcnt;
            if ((i__ret&0x8000)!=0)
                {
                ox8020_err=iberr;
                return sample_counter;
                }
            }
         /* Set EOS = 'LF' */
         ibeos (bd, 0x1C0A);
         }
    else /* RS232 */
         {
         for (i=0;i<count;i++)
              {
              target[i]=ComRdByte (ox8020_port);
              if ((rs232err)!=0)
                {
                ox8020_err=rs232err;
                return count;
                }
              }
         sample_counter=count;
         }

 /* **** Find X and Y scaling informations and Y offset in header **** */
if (head==1)
    {
    /* **** Look for X SCALE ***** */
    i__ret=FindPattern (ox8020_header, 0, -1, "SCALE", 0, 0);
    FillBytes (chaine, 0, 50, 0);
    j=0;
    for (i=i__ret+6;ox8020_header[i]!=' ';i++)
        {
        chaine[j]=ox8020_header[i];
        j++;
        }
    i__ret = Scan (chaine,"%s>%f",&ox8020_xscale);

    /* ***** Look for Y SCALE ***** */
    i__ret=FindPattern (ox8020_header, i, -1, "SCALE", 0, 0);
    FillBytes (chaine, 0, 50, 0);
    j=0;
    for (i=i__ret+6;ox8020_header[i]!=' ';i++)
        {
        chaine[j]=ox8020_header[i];
        j++;
        }
    i__ret = Scan (chaine,"%s>%f",&ox8020_yscale);

    /* ****** Look for Y OFFSET ******* */
    i__ret=FindPattern (ox8020_header, i, -1, "OFFSET", 0, 0);
    FillBytes (chaine, 0, 50, 0);
    j=0;
    for (i=i__ret+7;ox8020_header[i]!=' ';i++)
        {
        chaine[j]=ox8020_header[i];
        j++;
        }
    i__ret = Scan (chaine,"%s>%f",&ox8020_yoffset);

    /* ****** Look for SIZE  ******* */
    i__ret=FindPattern (ox8020_header, 0, -1, "SIZE", 0, 0);
    FillBytes (chaine, 0, 50, 0);
    j=0;
    for (i=i__ret+5;ox8020_header[i]!=')';i++)
        {
        chaine[j]=ox8020_header[i];
        j++;
        }
    i__ret = Scan (chaine,"%s>%i",&ox8020_size);
    }

    /* ***** Convert and store samples in array ************ */


    if ((target[0]!='4')||((target[0]=='4')&&(target[1]=='0')))
        {
        sample_counter-=9; /* Data - 'xxxx' - ')))) ' */
        i_start=4;
        }
    else
        {
        sample_counter-=10; /* Data - '4xxxx' - ')))) ' */
        i_start=5;
        }
        /* Convert samples */
        for (i=i_start;i<sample_counter+i_start;i++)
            {
            i__ret = Fmt (&itemp,"%i[u]<%c",target[i]);
            if (head==0)
                array[i-i_start] = (double)itemp;
            else
                array[i-i_start] = (double)((itemp - ox8020_yoffset) * ox8020_yscale);

            }
        count=sample_counter;

  if (interface)
	  SetXMode (ox8020_port, 1);       /* protocole XON/XOFF on */

  if (file==1)
    {
    /* Open File */
    file_id = OpenFile (filename, 2, 0, 1);
    /* Write only, truncate, ASCII */
    if (file_id ==  -1)
        {
        /* error */
        ox8020_err =  -8;
        return (0);
        }
    /* lecture des points dans le buffer : array */
    ic = 0;
    if (head==1)
        {
        FmtFile (file_id, "%s<%s\n", ox8020_header);
        }
    while (ic < ox8020_npoint)
        {
        Scan (&array[ic], "%f>%s",tampon);
        FmtFile (file_id, "%s<%s\n", tampon);
        ic += 1;
        }
    if (head==1)
        {
        FmtFile (file_id, "%s<%s\n", "))))");
        }
    /* fermeture du fichier */
    CloseFile (file_id);
    }
return (count);
}
int ox8020_check_cat()
{
    int env,i_ret_h,i_ret_l,i__ret;
    char chaine[5];

    flag_ch1AHL=0;
    flag_ch1BHL=0;
    flag_ch2AHL=0;
    flag_ch2BHL=0;
    FillBytes (chaine, 0, 5, 0);

    /* Read ACQuire:PEAK */
    if (ox8020_instrument_query(3,1)!=0)
        return ox8020_err;

    if ((ox8020_StrNCmp(ox8020_state,"OFF",3)==0))
        env=0;
    else
        env=1;

    /* if "ACQUIRE:PEAK OFF" then "ACQUIRE:PEAK ON" */
    if (env==0)
        ox8020_write_data ("ACQ:PEAK ON\n");

    /* Read CAT */
    ox8020_write_data ("TRAC:CAT?\n");
    if (!ox8020_err)
        {
        FillBytes (ox8020_state, 0, 60, 0);
        i__ret = ox8020_read_data (ox8020_state, 50);
        }

    /* Look for L in the CAT */
    i__ret=0;
    while (i__ret >= 0)
        {
        i__ret=FindPattern (ox8020_state, i__ret, -1, "L", 0, 0);
        if (i__ret>0)
            {
            if (ox8020_instrument==8020)
                {
                /* CH1 or CH2 ? */
                if (ox8020_state[i__ret-1]=='1')
                    flag_ch1AHL=1;
                else
                    if (ox8020_state[i__ret-1]=='2')
                        flag_ch2AHL=1;
                }
            else
                {
                /* CH1A or CH1B or CH2A or CH2B ? */
                chaine[0]=ox8020_state[i__ret-2];
                chaine[1]=ox8020_state[i__ret-1];
                if ((ox8020_StrNCmp(chaine,"1A" , 2))==0)
                    flag_ch1AHL=1;
                else
                    if ((ox8020_StrNCmp(chaine,"1B" , 2))==0)
                        flag_ch1BHL=1;
                    else
                        if ((ox8020_StrNCmp(chaine,"2A" , 2))==0)
                            flag_ch2AHL=1;
                        else
                            if ((ox8020_StrNCmp(chaine,"2B" , 2))==0)
                                flag_ch2BHL=1;
                }
            i__ret++;
            }

        }
    /* Look for S2 in the CAT */
    i__ret=FindPattern (ox8020_state, 0, -1, "S2", 0, 0);
    if (i__ret==-1)  flag_visu_s2=0;
    else            flag_visu_s2=1;

    if (env==0)
        ox8020_write_data ("ACQ:PEAK OFF\n");

return ox8020_err;
}

/* ------------------------------------------------- */
/* Routine name :   measure                          */
/* Function     :   Read a measure on the instrument */
/* ------------------------------------------------- */
int ox8020_measure (control, source)
int control;
int source;
{
    int n;
    char cmd[200];
    int entier;
    double reel;
    int i__ret;

    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if (( -(control < 0) |  -(control > 16)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if (ox8020_instrument==8020)
        if ((source<1 )||(source>2))
            {
            ox8020_err =  -2;
            return ox8020_err;
            }
    else
        if ((source<1 )||(source>4))
            {
            ox8020_err =  -2;
            return ox8020_err;
            }
    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    FillBytes (cmd, 0, 200, 0);

    switch (control)  {
    case 0:
        n = Fmt (cmd, "%s<MEAS:AC? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 1:
        n = Fmt (cmd, "%s<MEAS:DC? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 2:
        n = Fmt (cmd, "%s<MEAS:FREQ? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 3:
        n = Fmt (cmd, "%s<MEAS:PER? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 4:
        n = Fmt (cmd, "%s<MEAS:PHAS? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 5:
        n = Fmt (cmd, "%s<MEAS:AMPL? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 6:
        n = Fmt (cmd, "%s<MEAS:LOW? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 7:
        n = Fmt (cmd, "%s<MEAS:HIGH? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 8:
        n = Fmt (cmd, "%s<MEAS:RTIME? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 9:
        n = Fmt (cmd, "%s<MEAS:FTIME? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 10:
        n = Fmt (cmd, "%s<MEAS:NWID? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 11:
        n = Fmt (cmd, "%s<MEAS:PWID? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 12:
        n = Fmt (cmd, "%s<MEAS:PDUT? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 13:
        n = Fmt (cmd, "%s<MEAS:NDUT? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 14:
        n = Fmt (cmd, "%s<MEAS:MIN? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 15:
        n = Fmt (cmd, "%s<MEAS:MAX? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    case 16:
        n = Fmt (cmd, "%s<MEAS:PTP? (@%d)\n", source);
        i__ret = ox8020_write_data (cmd);
        break;
    default:
        break;
    }
    if (!ox8020_err)
        {
        FillBytes (ox8020_mesure, 0, 20, 0);
        i__ret = ox8020_read_data (ox8020_mesure, 20);
        /* Bug Versions before V1.4 */
        i__ret = FindPattern (ox8020_mesure, 0, -1, "{", 0, 0);
        if (i__ret!=-1)
            ox8020_mesure[i__ret]=0;
        }
    if (ox8020_err!=0)
        FillBytes (ox8020_mesure, 0, 20, 0);

return ox8020_err;
}


/* ------------------------------------------------- */
/* Routine name :   status                           */
/* Function     :   Read status of instrument        */
/* ------------------------------------------------- */
int ox8020_status (status_num,mask)
int     status_num;
int     mask;
{
    int i__ret;
    char cmd[100];

    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if (( -(status_num < 0) |  -(status_num > 9)) != 0)  {
        ox8020_err =  -1;
        return ox8020_err;
    }
    if ((mask < 0) ||  (mask > 255)) {
        ox8020_err =  -2;
        return ox8020_err;
    }
    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    switch (status_num)  {
    case 0:
        ox8020_write_data ("*ESE?\n");
        break;
    case 1:
        ox8020_write_data ("*ESR?\n");
        break;
    case 2:
        ox8020_write_data ("*IDN?\n");
        break;
    case 3:
        /* Disable  XON/XOFF (Binary Data) */
    	if (interface)
	        SetXMode (ox8020_port, 0);
        ox8020_write_data ("*LRN?\n");
        break;
    case 4:
        ox8020_write_data ("*OPC?\n");
        break;
    case 5:
        ox8020_write_data ("*SRE?\n");
        break;
    case 6:
        ox8020_write_data ("*STB?\n");
        break;
    case 7:
        ox8020_write_data ("*TST?\n");
        break;
    case 8:
        Fmt (cmd, "%s<*ESE %d\n", mask);
        ox8020_write_data (cmd);
        break;
    case 9:
        Fmt (cmd, "%s<*SRE %d\n", mask);
        ox8020_write_data (cmd);
        break;
    default:
        break;
    }
    if ((!ox8020_err)&&(status_num<8))
        {
        FillBytes (ox8020_state, 0, 60, 0);
        if (status_num!=3)
            i__ret = ox8020_read_data (ox8020_state,50);
        else
            {
            i__ret = ox8020_read_data (ox8020_state,57 );
            /* Enable  XON/XOFF  */
            if (interface)
				SetXMode (ox8020_port, 1);
            }
        }
    if (ox8020_err!=0)
        FillBytes (ox8020_state, 0, 60, 0);

return(ox8020_err);
}


/* ------------------------------------------------- */
/* Routine name :   instrument_query                 */
/* Function     :   Instrument Query                 */
/* ------------------------------------------------- */
int ox8020_instrument_query(query,channel)
int query;
int channel;
{
    int i_ret;
    char *query_array[33];
    char cmd[100];

    /* Query array initialization */
    query_array[0]  = "MODE?\n";
    query_array[1]  = "ACQ:GLIT?\n";
    query_array[2]  = "ACQ:MODE?\n";
    query_array[3]  = "ACQ:PEAK?\n";
    query_array[4]  = "ACQ:REF?\n";
    query_array[5]  = "DISP:ANN?\n";
    query_array[6]  = "DISP:FILT?\n";
    query_array[7]  = "DISP:MODE?\n";
    query_array[8]  = "DISP:SAVE";
    query_array[9]  = "DISP:TRAC:FEED?\n";
    query_array[10] = "DISP:TRAC:X:LEFT?\n";
    query_array[11] = "DISP:TRAC:X:PDIV?\n";
    query_array[12] = "FORM?\n";
    query_array[13] = "FORM:DINT?\n";
    query_array[14] = "INP1:COUP?\n";
    query_array[15] = "INP1:RANG?\n";
    query_array[16] = "INP1:VERN?\n";
    query_array[17] = "INP2:COUP?\n";
    query_array[18] = "INP2:INV?\n";
    query_array[19] = "INP2:RANG?\n";
    query_array[20] = "INP2:VERN?\n";
    query_array[21] = "OUTP:FILT?\n";
    query_array[22] = "SYST:ERRor?\n";
    query_array[23] = "SYST:KLOck?\n";
    query_array[24] = "SYST:VERS?\n";
    query_array[25] = "TIM:DEL?\n";
    query_array[26] = "TIM:EXPH?\n";
    query_array[27] = "TIM:MODE?\n";
    query_array[28] = "TIM:RANG?\n";
    query_array[29] = "TRIG:COUP?\n";
    query_array[30] = "TRIG:MODE?\n";
    query_array[31] = "TRIG:SLOP?\n";
    query_array[32] = "TRIG:SOUR?\n";

    ox8020_err = 0;
    /* **************** Parameter validation ************* */
    if ((query<0)||(query>32))
        {
        ox8020_err =  -1;
        return ox8020_err;
        }

    if ((channel<1)||(channel > 4))
        {
        ox8020_err =  -2;
        return ox8020_err;
        }
    /* *************************************************** */
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    if (ox8020_instrument==8020)
        {
        switch (channel)
            {
            case 1:
            case 2:
                channel=1;
                break;
            case 3:
            case 4:
                channel=2;
                break;
            default:
                break;
            }
        }

    FillBytes (ox8020_state, 0, 60, 0);

    switch (query)
        {
        case 8:         /* "DISP:SAVE<N>?\n" */
                    Fmt (cmd,"%s<%s%d%s",query_array[query], channel,"?\n");
                    ox8020_write_data (cmd);
                    break;
       default:
                    ox8020_write_data (query_array[query]);
                    break;
        }
    if (!ox8020_err)
        i_ret = ox8020_read_data (ox8020_state, 50);
    else
        FillBytes (ox8020_state, 0, 60, 0);

return(ox8020_err);
}

/* ------------------------------------------------- */
/* Routine name :   close                            */
/* Function     :   close instrument                 */
/* ------------------------------------------------- */
int ox8020_close (void)
{

    ox8020_err = 0;
    if (ox8020_device_closed () != 0)
        return ox8020_err;

    /* Unlock Front Pannel */
    if (flag_klock==1)
        ox8020_write_data ("SYST:KLOCK OFF\n");

    if (!interface)
        {
        if (ibloc (bd) <= 0)
            {
            ox8020_err = 234;
            return ox8020_err;
            }
        if (CloseDev (bd) < 0)
            ox8020_err = 221;
        }
    else
        {
        /* Wait for the output buffer empty */
        while (GetOutQLen (ox8020_port)!=0) {}

        CloseCom(ox8020_port);
        if (rs232err != 0)
            {
            ox8020_err = rs232err+300;
            return ox8020_err;
            }
        ox8020_port=0;
        }

    bd = 0;
    ox8020_port = 0;
    return ox8020_err;
}


/* ------------------------------------------------- */
/* Routine name :   read_data                        */
/* Function     :   Read data String on GPIB or RS232*/
/* ------------------------------------------------- */
int ox8020_read_data (echo, cnt)
char *echo;
int cnt;
{
    int i_ret,i;

    if (!interface)  /* IEEE */
        {
        if (ibrd (bd, echo, (long)cnt) <= 0)
            ox8020_err = 231;
        else
            {
            ox8020_err = 0;
            }
        }
    else
        {          /* RS232 */
        /* Bug with LW CVI 3.0.1                            */
        /* i_ret = ComRdTerm (ox8020_port, echo, cnt, NL);  */
        /* Replace with :                                   */
        i=0;
        echo[i]=ComRdByte (ox8020_port);
        if (rs232err != 0)
            {
            ox8020_err = 231;
            return ox8020_err;
            }
        /* if SAVE Instrument Setup , dot not terminate on NL (Binary Data) */
        if (cnt==57)
            {
            while ((i<cnt-1)&&(rs232err==0))
                {
                i++;
                echo[i]=ComRdByte (ox8020_port);
                }
            }
        else
            {
            while ((echo[i]!=10)&&(i<cnt-1)&&(rs232err==0))
                {
                i++;
                echo[i]=ComRdByte (ox8020_port);
                }
            }
        /* ************************************************ */
        if (rs232err != 0)
            ox8020_err = 231;
        else
            ox8020_err =0;
        }
return ox8020_err;
}


/* -------------------------------------------------- */
/* Routine name :   write_data                        */
/* Function     :   Write String data on GPIB or RS232*/
/* -------------------------------------------------- */
int ox8020_write_data (cmd)
char *cmd;
{
    int n;
    int i__ret;

   if (!interface)  /* ******** IEEE ******** */
        {
        if (ibwrt (bd, cmd, (long)StringLength (cmd)) <= 0)
            ox8020_err = 230;
        else
            ox8020_err = 0;

        }
   else            /* ******** RS232 ******** */
        {
        ComWrt (ox8020_port, cmd, StringLength (cmd));
        if (rs232err  !=0)
            ox8020_err = 230;
        else
            ox8020_err = 0;
        }
return ox8020_err;
}


/* ------------------------------------------------------------------- */
/* Routine name :  device_closed                                       */
/* Function     :  This function checks to see if the module has been  */
/*                 initialized.  If the device has not been opened, a 1*/
/*                 is returned, 0 otherwise.                           */
/* ------------------------------------------------------------------- */
int ox8020_device_closed ()
{
    if (interface==0)
        {
        if (bd <= 0)
            {
            ox8020_err = 232;
            return -1;
            }
        return 0;
        }
    else
        {
        if (ox8020_port == 0)
            {
            ox8020_err = 232;
            return -1;
            }
        return 0;
        }
}
/* ------------------------------------------------- */
/* Routine name :   ox8020_StrNCmp                          */
/* Function     :   Compare N characters in 2 strings*/
/* ------------------------------------------------- */
int ox8020_StrNCmp (src1, src2, l__lg)
char *src1;     /* First string to be compared */
char *src2;     /* Second string to be compared */
int l__lg;      /* Byte number to be compared */
{
    int i;
    int ll;
    int l_min;
    int res;
    int lg;

    /*  - Modified value parameter */
    lg = l__lg;
    ll = StringLength (src1);
    l_min = StringLength (src2);
    if (ll < l_min)
        l_min = ll;
    if (( -(lg ==  -1) |  -(lg > l_min)) != 0)
        lg = l_min;
    res = 0;
    i = 0;
    while (( -(i < lg) &  -(!res)) != 0)  {
        if (src1[i] < src2[i])
            res =  -1;
        else  if (src1[i] > src2[i])
            res = 1;
        i++;
    }
    if (!res)
        if (StringLength (src1) < StringLength (src2))
            res = 1;
        else  if (StringLength (src1) > StringLength (src2))
            res =  -1;
        else
            res = 0;
    return res;
}
/* *********** End of LabWindows 2.3 OX8020/OX8027 Driver **************** */

